defmodule Reseau.Etape.NomJoueur do
  @moduledoc """
  Une étape pour demander le nom du joueur à l'utilisateur.
  """

  @behaviour Reseau.Etape

  @typedoc "Un joueur"
  @type joueur() :: Reseau.Joueur.t()

  @typedoc "Une partie"
  @type partie() :: Reseau.Partie.t()

  @impl true
  @doc """
  Callback appelée quand l'étape est chargée et prête à être affichée.

  Arguments :

  * `joueur` : le joueur actuel (`Reseau.Joueur`) ;
  * `partie` : la partie actuelle (`Reseau.Partie`).

  Elle doit retourner un tuple contenant le joueur et la partie (éventuellement modifiés).

  """
  @spec entrer(joueur(), partie()) :: {joueur(), partie()}
  def entrer(joueur, partie), do: {joueur, partie}

  @impl true
  @doc """
  Retourne le titre de l'étape.
  """
  @spec titre(joueur(), partie()) :: String.t()
  def titre(_joueur, _partie), do: "Nom du joueur"

  @impl true
  @doc """
  Retourne le texte de l'étape.
  """
  @spec texte(joueur(), partie()) :: String.t()
  def texte(_joueur, _partie) do
    """
    Entrez le nom de votre joueur. Ce nom ne sera utilisé que pour vous identifier
    auprès des autres joueurs connectés à la plateforme.
    Le nom doit contenir au minimum 3 lettres.
    """
  end

  @impl true
  @doc """
  Retourne le prompt de l'étape.
  """
  @spec prompt(joueur(), partie()) :: String.t()
  def prompt(_joueur, _partie), do: "Entrez le nom de votre joueur :"

  @impl true
  @doc """
  Gère les commandes entrées par l'utilisateur.

  La commande entrée est précisée sous la forme d'une chaîne de caractères. Le retour peut être de plusieurs types :

  - `:silence` : n'affiche rien et ne fait rien ;
  - `:prompt` : n'affiche rien sauf le prompt ;
  - `:rafraîchir` : n'affiche que l'écran de nouveau, sans message ;
  - `{:silence, message}` : affiche juste le message ;
  - `{:prompt, message}` : n'affiche que le message suivi du prompt ;
  - `{:rafraîchir, message}` : affiche le message suivi de tout l'écran ;
  - `{:rediriger, module, joueur, partie}` : change l'étape actuelle, affiche le nouvel écran ;
  - `{:rediriger, module, joueur, partie, message}` : change l'étape actuelle, affiche le message suivi du nouvel écran ;
  - `{:rediriger, module, joueur, partie, :silence}` : change l'étape actuelle, n'affiche rien ;
  - `{:rediriger, module, joueur, partie, :silence, message}` : change l'étape actuelle, affiche uniquement le message.

  """
  @spec gérer_commandes(joueur(), partie(), String.t()) ::
              :silence
              | :prompt
              | :rafrîchir
              | {:silence, String.t()}
              | {:prompt, String.t()}
              | {:rafraîchir, String.t()}
              | {:rediriger, module(), joueur(), partie()}
              | {:rediriger, module(), joueur(), partie(), String.t()}
              | {:rediriger, module(), joueur(), partie(), :silence}
              | {:rediriger, module(), joueur(), partie(), :silence, String.t()}
  def gérer_commandes(joueur, partie, commande) do
    if String.length(commande) < 3 do
      {:prompt, "Le nom du joueur doit contenir au moins 3 caractères."}
    else
      joueur = %{joueur | nom: commande}
      {:rediriger, Reseau.Etape.Parties, joueur, partie}
    end
  end
end
